#!/usr/bin/env bash
# stop script on error
set -e

# The files generated by this script are used in unit tests that create TLS
# connections between a localhost server and client. Run this script if any
# certificates used for tests are expired.

# Files generated by this script:
# ca_root.crt:        root certificate authority
# server.crt:         certificate for server signed by ca_root.crt
# server.key:         private key for server.crt
# server_chain.crt:   certificate chain containing server.crt and ca_root.crt
# unittests.crt:      self-signed certificate
# unittests.p8:       private key, pkcs#8 syntax
# unittests.p12:      pkcs#12 file bundling the certificate and private key. Password is "1234"
# ec_unittests.crt:   elliptic curve self-signed certificate
# ec_unittests.p8:    elliptic curve private key, pkcs#8 syntax
# ec_unittests.p12:   elliptic curve pkcs#12 file bundling the certificate and private key. Password is "1234"

# Create directory for use with certificate generation
mkdir -p certGeneration
# Copy files needed to generate new certificates
cp unittests.key certGeneration/unittests.key
cp ec_unittests.key certGeneration/ec_unittests.key
cp unittests.conf certGeneration/unittests.conf
cp ca_root.cnf certGeneration/ca_root.cnf
cd certGeneration

# index.txt and serial are required for use with openssl config files
touch index.txt
echo 1000 > serial

# Generating a key for the new ca_root
openssl genrsa -out ca_root.key 2048

# Generate the ca_root certificate
openssl req -config ca_root.cnf \
            -key ca_root.key \
            -new -x509 -days 824 -sha256 -extensions v3_ca \
            -out ca_root.crt \
            -set_serial 00 \
            -subj '/C=US/ST=Washington/L=Seattle/O=Amazon/OU=SDKs/CN=localhostCA/emailAddress=aws-sdk-common-runtime@amazon.com'

# Generate a private key for the server
openssl genrsa -out server.key 2048

# Generate a certificate signing request for the server
openssl req -new -sha256 \
            -key server.key \
            -out server.csr \
            -set_serial 02 \
            -subj '/C=US/ST=Washington/L=Seattle/O=Amazon/OU=SDKs/CN=localhost/emailAddress=aws-sdk-common-runtime@amazon.com'

# Sign the server signing request with ca_root
yes | openssl ca -config ca_root.cnf \
            -extensions server_cert \
            -days 824 -notext -md sha256 \
            -in server.csr \
            -out server.crt

# Generate a certificate chain containing the ca_root and server certificates
cat server.crt ca_root.crt > server_chain.crt

# Generate other unittest certificate variations
for base in unittests ec_unittests; do
  openssl req -x509 -new \
            -key $base.key \
            -config unittests.conf \
            -out $base.crt \
            -days 824

  openssl pkcs8 -topk8 \
            -out $base.p8 \
            -in $base.key \
            -nocrypt

  openssl pkcs12 -export \
            -out $base.p12 \
            -inkey $base.key \
            -in $base.crt \
            -password pass:1234
done

# Copy the generated certificates and keys to the resources folder
cd ..
cp certGeneration/ca_root.crt ./ca_root.crt
cp certGeneration/server.crt ./server.crt
cp certGeneration/server.key ./server.key
cp certGeneration/server_chain.crt ./server_chain.crt
cp certGeneration/server.crt ./server.crt

for base in unittests ec_unittests; do
  cp certGeneration/$base.crt ./$base.crt
  cp certGeneration/$base.p8 ./$base.p8
  cp certGeneration/$base.p12 ./$base.p12
done

# Clean up the certGeneration folder
rm -r certGeneration
